/******************************************************************************
* (C) Copyright 2002 by Agilent Technologies, All Rights Reserved.
******************************************************************************/

#include <string.h>

#include "AgtPCIEExerciser.h"
#include "AgtPCIEControl.h"
#include "devpcieintdefs.h"
#include "PCIEMemoryMgmt.h"
#include "PCIEFPGAFramework.h" 
#include "PCIESiMgmt.h"

#pragma region Macros

// Check, wether offset into configspace is allowed (DW-aligned)
#define CONFREGOFFSETCHECK( offset ) ( offset & 0x3 ? 1 : 0 )

// This is not implemented
#define BLOCKMEMORYSTOPCHECK \
  {\
    if( mBlockMemoryStopped != 1 ) \
    AGT_THROW( "Error: Block memory not stopped" ); \
  }

#define BUFFERCHECKREAD \
  {\
    if( *pCount != length ) \
    {\
      char buf[NUMBER_OF_CHAR_CHARACTERS_FOR_ERR_MSG]; \
      sprintf( buf, "The parameter 'pCount' needs to contain the number of bytes allocated for 'psaData'. Please provide a buffer of %d bytes.\n", length ); \
      AGT_THROW( buf ); \
    }\
  }

#define BUFFERCHECKWRITE \
  {\
    if( numBytes % 5 != 0 ) \
    {\
      char buf[NUMBER_OF_CHAR_CHARACTERS_FOR_ERR_MSG]; \
      sprintf(buf, "The parameter 'numBytes' needs to contain the number of bytes allocated for 'psaData'. It needs to be 'numBytes modulo 5 = 0'.\n" ); \
      AGT_THROW( buf ); \
    }\
  }

#define TRACE(msg)

#define NUMBER_OF_CHAR_CHARACTERS_FOR_ERR_MSG ((int)256)

#pragma endregion

#pragma region Constructor and Destructor

CAgtPCIEExerciser::CAgtPCIEExerciser( void )
{
  myController = new CAgtPCIEControl();       // returns only pointer if instance exists
  myFpgaFramework = new CPCIEFpgaFramework(); // will access controller
  myMemoryMgmt = new CPCIEMemoryMgmt();       // will access FPGAFramework
  mSiMgmt = new CPCIESiMgmt();
  mBlockMemoryStopped = 1;
  mReleaseFPGAAfterEachCall = 0;              // Default false(0)
}

CAgtPCIEExerciser::~CAgtPCIEExerciser( void )
{
  delete mSiMgmt;
  delete myMemoryMgmt;
  delete myFpgaFramework;
  delete myController;

  myController = NULL;
  myMemoryMgmt = NULL;
  myFpgaFramework = NULL;
  mSiMgmt = NULL;
}

#pragma endregion

#pragma region Helpers functions

#pragma region Public functions

//void CAgtPCIEExerciser::FPGARegRead
//(
//  /* [in] */ AgtPortHandleT portHandle,
//  /* [in] */ UInt32 address,
//  /* [in] */ UInt8 size,
//  /* [retval][out] */ UInt32& val
//)
//{
//  if( myController->IsISFPGAEnabled( portHandle ) )
//  {
//    myController->GetFunctionLockAndFPGAAccess( portHandle );
//
//    myController->FPGARead( portHandle, address, val );
//
//    if( mReleaseFPGAAfterEachCall )
//      myController->FPGAAccessRelease( portHandle );
//  }
//}
//
//void CAgtPCIEExerciser::FPGARegRead
//(
//  /* [in] */ AgtPortHandleT portHandle,
//  /* [in] */ UInt32 address,
//  /* [in] */ UInt8 size,
//  /* [retval][out] */ UInt32* val
//)
//{
//  if( val == NULL )
//    AGT_THROW( "Invalid NULL pointer passed to FPGARegRead()." );
//
//  FPGARegRead( portHandle, address, size, *val );
//}

void CAgtPCIEExerciser::RegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [retval][out] */ UInt32& val
)
{
  RegRead( portHandle, address, size, false, false, val );
}

void CAgtPCIEExerciser::RegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [retval][out] */ UInt32* val
)
{
  RegRead( portHandle, address, size, false, false, val );
}

void CAgtPCIEExerciser::RegWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [in] */ UInt32 val
)
{
  RegWrite( portHandle, address, size, false, false, val );
}

void CAgtPCIEExerciser::RegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [in] */ bool bDoNotLock,
  /* [in] */ bool bDoNotUnlock,
  /* [retval][out] */ UInt32& val
)
{
  myController->RegRead( portHandle, address, size, bDoNotLock, bDoNotUnlock, val );
}

void CAgtPCIEExerciser::RegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [in] */ bool bDoNotLock,
  /* [in] */ bool bDoNotUnlock,
  /* [retval][out] */ UInt32* val
)
{
  if( val == NULL )
    AGT_THROW( "NULL pointer passed to RegRead()." );

  UInt32 value = 0;

  myController->RegRead( portHandle, address, size, bDoNotLock, bDoNotUnlock, value );

  *val = value;
}

void CAgtPCIEExerciser::RegWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt32 address,
  /* [in] */ UInt8 size,
  /* [in] */ bool bDoNotLock,
  /* [in] */ bool bDoNotUnlock,
  /* [in] */ UInt32 val
)
{
  myController->RegWrite( portHandle, address, size, bDoNotLock, bDoNotUnlock, val );
}

void CAgtPCIEExerciser::SetFunctionNumber
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ UInt8 functionNum
)
{
  myController->SetFunctionNumber( portHandle, functionNum );
}

void CAgtPCIEExerciser::StartDebugLogging
(
  /* [in] */ char* strFileName_p
)
{
  myController->StartDebugLogging( strFileName_p );
}

#pragma endregion

#pragma region Protected functions

UInt32 CAgtPCIEExerciser::getDecTypePropertyAddress
(
  EPCIEDec dec,
  EPCIEDecProp prop
)
{
  UInt32 baseAddress;

  baseAddress = ( ( UInt32 ) dec );
  baseAddress *= PCIE_CONFIGSPACE_PROP_SIZE;
  baseAddress +=( ( UInt32 ) prop );

  return baseAddress;
}

void CAgtPCIEExerciser::MemWrite
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 numBytes,
  UInt32 size,
  UInt8* psaData
)
{
  myController->RegBlockWrite( portHandle, address, numBytes, size, psaData );
}

void CAgtPCIEExerciser::MemRead
(
  AgtPortHandleT portHandle,
  UInt32 address,
  UInt32 numBytes,
  UInt32 size,
  UInt32* pCount,
  UInt8* psaData
)
{
  myController->RegBlockRead( portHandle, address, numBytes, size, pCount, psaData );
}

void CAgtPCIEExerciser::GetBlockPropAddress
(
  EPCIEHwChannelFunction channel,
  AgtValueT lineIndex,
  AgtValueT* address
)
{
  if( lineIndex >= PCIE_BLOCKLINE_NUM )
  {
    AGT_THROW( "GetBlockPropAddress: Parameter lineIndex must be less than PCIE_BLOCKLINE_NUM" );
  }

  switch( channel )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    *address = PCIE_BLOCKLINESTART_HWCHANNELA_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    *address = PCIE_BLOCKLINESTART_HWCHANNELB_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    *address = PCIE_BLOCKLINESTART_HWCHANNELC_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    *address = PCIE_BLOCKLINESTART_HWCHANNELD_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    *address = PCIE_BLOCKLINESTART_HWCHANNELE_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    *address = PCIE_BLOCKLINESTART_HWCHANNELBVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    *address = PCIE_BLOCKLINESTART_HWCHANNELBVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    *address = PCIE_BLOCKLINESTART_HWCHANNELCVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    *address = PCIE_BLOCKLINESTART_HWCHANNELCVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    *address = PCIE_BLOCKLINESTART_HWCHANNELDVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    *address = PCIE_BLOCKLINESTART_HWCHANNELDVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    *address = PCIE_BLOCKLINESTART_HWCHANNELEVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    *address = PCIE_BLOCKLINESTART_HWCHANNELEVF2_ADDR + lineIndex;
    break;
  default:
    AGT_THROW( "GetBlockPropAddress: Parameter channel has invalid value" );
  }
}

void CAgtPCIEExerciser::GetReqBehPropAddress
(
  EPCIEHwChannelFunction channel,
  AgtValueT lineIndex,
  AgtValueT* address
)
{
  if( lineIndex >= PCIE_REQBEHLINE_NUM )
  {
    AGT_THROW( "GetBlockPropAddress: Parameter lineIndex must be less than PCIE_REQBEHLINE_NUM" );
  }

  switch( channel )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    *address = PCIE_REQBEHLINESTART_HWCHANNELA_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    *address = PCIE_REQBEHLINESTART_HWCHANNELB_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    *address = PCIE_REQBEHLINESTART_HWCHANNELC_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    *address = PCIE_REQBEHLINESTART_HWCHANNELD_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    *address = PCIE_REQBEHLINESTART_HWCHANNELE_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    *address = PCIE_REQBEHLINESTART_HWCHANNELBVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    *address = PCIE_REQBEHLINESTART_HWCHANNELBVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    *address = PCIE_REQBEHLINESTART_HWCHANNELCVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    *address = PCIE_REQBEHLINESTART_HWCHANNELCVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    *address = PCIE_REQBEHLINESTART_HWCHANNELDVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    *address = PCIE_REQBEHLINESTART_HWCHANNELDVF2_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    *address = PCIE_REQBEHLINESTART_HWCHANNELEVF1_ADDR + lineIndex;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    *address = PCIE_REQBEHLINESTART_HWCHANNELEVF2_ADDR + lineIndex;
    break;
  default:
    AGT_THROW( "GetReqBehPropAddress: Parameter channel has invalid value" );
  }
}

void CAgtPCIEExerciser::GetCompBehPropAddress
(
  EPCIECompQueue queue,
  AgtValueT lineIndex,
  AgtValueT* address
)
{
  if( lineIndex >= PCIE_COMPBEHLINE_NUM )
  {
    AGT_THROW( "GetCompBehPropAddress: Parameter lineIndex must be less than PCIE_COMPBEHLINE_NUM" );
  }

  switch( queue )
  {
  case PCIE_COMPQUEUE_0:
    *address = PCIE_COMPBEHLINESTART_QUEUEA_ADDR + lineIndex;
    break;
  case PCIE_COMPQUEUE_1:
    *address = PCIE_COMPBEHLINESTART_QUEUEB_ADDR + lineIndex;
    break;
  case PCIE_COMPQUEUE_2:
    *address = PCIE_COMPBEHLINESTART_QUEUEC_ADDR + lineIndex;
    break;
  case PCIE_COMPQUEUE_3:
    *address = PCIE_COMPBEHLINESTART_QUEUED_ADDR + lineIndex;
    break;
  case PCIE_COMPQUEUE_4:
    *address = PCIE_COMPBEHLINESTART_QUEUEE_ADDR + lineIndex;
    break;
  default:
    AGT_THROW( "GetCompBehPropAddress: Parameter queue has invalid value" );
  }
}

bool CAgtPCIEExerciser::ExerciserStatusReadFunction
(
  AgtPortHandleT portHandle,
  EPCIEExerciserStatus status,
  AgtValueT* val
)
{
  bool returnVal = false;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    returnVal = true; // per default return a 'call executed' status
    *val = 0;

    switch( status )
    {
    case PCIE_EXERCISERSTATUS_HWCHANNEL_A_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_A );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_B_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_B );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_C_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_C );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_D_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_D );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_E_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_E );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_BVF1_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_BVF1 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_BVF2_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_BVF2 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_CVF1_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_CVF1 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_CVF2_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_CVF2 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_DVF1_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_DVF1 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_DVF2_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_DVF2 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_EVF1_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_EVF1 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_HWCHANNEL_EVF2_STATE:
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = myMemoryMgmt->GetHWChannelState( portHandle, PCIE_HWCHANNEL_FUNCTION_EVF2 );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
      break;
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_DATA_EXPECTED: // is prepared for ISP FPGA, test needed (\spt_pcie_gen2_in_system\25)
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_DATA_REAL: // is prepared for ISP FPGA, test needed (\spt_pcie_gen2_in_system\25)
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_ERROR_OCCURRED:  // is prepared for ISP FPGA, test needed (\spt_pcie_gen2_in_system\25)
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_ERROR_COUNT: // is prepared for ISP FPGA, test needed (\spt_pcie_gen2_in_system\25)
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_A_INTADDR: // is prepared for ISP FPGA, test needed (\spt_pcie_gen2_in_system\25)
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_B_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_C_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_D_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_E_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF1_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_BVF2_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF1_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_CVF2_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF1_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_DVF2_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF1_INTADDR:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_DATA_EXPECTED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_DATA_REAL:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_ERROR_OCCURRED:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_ERROR_COUNT:
    case PCIE_EXERCISERSTATUS_DATAMEMCMP_HWCHANNEL_EVF2_INTADDR:
    case PCIE_EXERCISERSTATUS_HARDWARE:
    case PCIE_EXERCISERSTATUS_REVERSED_LANES: 
    case PCIE_EXERCISERSTATUS_RX_POLARITY_INVERSION: 
    case PCIE_EXERCISERSTATUS_OBSOLETE_09: // this is an invalid enum entry; this is documented as such
    case PCIE_EXERCISERSTATUS_OBSOLETE_00: // obsolete
    case PCIE_EXERCISERSTATUS_OBSOLETE_01: // obsolete
    case PCIE_EXERCISERSTATUS_OBSOLETE_02: // obsolete
    case PCIE_EXERCISERSTATUS_OBSOLETE_03: // obsolete
    case PCIE_EXERCISERSTATUS_OBSOLETE_04: // obsolete
    case PCIE_EXERCISERSTATUS_OBSOLETE_05: // obsolete
    case PCIE_EXERCISERSTATUS_OUTSTANDING_REQUESTS: 
    case PCIE_EXERCISERSTATUS_INVALID_CLOCK: 
    case PCIE_EXERCISERSTATUS_PROTERR:
    case PCIE_EXERCISERSTATUS_LTSSM_STATE: 
    case PCIE_EXERCISERSTATUS_LTSSM_VERSION: 
    case PCIE_EXERCISERSTATUS_LINKSTATE_DIRECT_PENDING: 
    case PCIE_EXERCISERSTATUS_POWERSTATE: 
    case PCIE_EXERCISERSTATUS_LINKSPEED: 
    case PCIE_EXERCISERSTATUS_LAST_TRAINING_CONTROL: 
    case PCIE_EXERCISERSTATUS_DATARATE_RECEIVED: 
    case PCIE_EXERCISERSTATUS_DATARATE_ADVERTISED: 
    case PCIE_EXERCISERSTATUS_UPCONFIGURE_CAPABILITY_RECEIVED: 
    default:
      returnVal = false;
    }
  }

  return returnVal;
}

#pragma endregion

#pragma endregion

#pragma region API functions

#pragma region Link / Phy Control

void CAgtPCIEExerciser::DllPhySet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllPhy prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllPhySet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllPhy, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::DllPhySet" );

  return RegWrite( portHandle, PCIE_DLLPHY_PROP_ADDR + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::DllPhyGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllPhy prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllPhyGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllPhy, prop );

  TRACE( "CAgtPCIEExerciser::DllPhyGet" );

  return RegRead( portHandle, PCIE_DLLPHY_PROP_ADDR + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::LinkStateDirect
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIELinkStateDirect state
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "LinkStateDirect",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIELinkStateDirect, state );

  TRACE( "CAgtPCIEExerciser::LinkStateDirect" );

  AgtValueT lastDirectSuccessful;

  ExerciserStatusRead( portHandle, PCIE_EXERCISERSTATUS_LINKSTATE_DIRECT_PENDING, &lastDirectSuccessful );
  
  // the last direct link command has not been successful (i.e. at least on of the direct
  // link bits is 1 AND the targetState is not 'IDLE' or 'RETRAIN' we do nothing
  if( lastDirectSuccessful == 1 &&
      state != PCIE_LINKSTATEDIRECT_IDLE &&
      state != PCIE_LINKSTATEDIRECT_START_LINKTRAINING )
  {
    //return E_AGT_PCIEEXERCISER_LINKSTATE_DIRECT_PENDING;
  }
  else
    return RegWrite( portHandle, PCIE_LINKSTATE_DIRECT_ADDR, 32, ( UInt32 )state );
}

void CAgtPCIEExerciser::LinkUp
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "LinkUp",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::LinkUp" );

  return RegWrite( portHandle, PCIE_LINKUP_ADDR, 32, 0x0 );
}

#pragma endregion

#pragma region ExerciserGet, ExerciserSet

void CAgtPCIEExerciser::ExerciserSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEExerciser prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEExerciser, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ExerciserSet" );

  return RegWrite( portHandle, PCIE_EXERCISER_ADDR + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::ExerciserGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEExerciser prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEExerciser, prop );

  TRACE( "CAgtPCIEExerciser::ExerciserGet" );

  return RegRead( portHandle, PCIE_EXERCISER_ADDR + ( UInt32 )prop, 32, val );
}

#pragma endregion

#pragma region Traffic Setup

#pragma region Start / Stop

void CAgtPCIEExerciser::Run
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "Run",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::Run" );

  // This should not be done using FPGA as some settings like function enabled, mode are only written
  // when run is called. May cause inconsistant behavior when using multiple functions.

  /*if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->RequesterRun( portHandle );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {*/
    RegWrite( portHandle, PCIE_RUN_ADDR, 32, 0x0 );
  /*}*/
}

void CAgtPCIEExerciser::Stop
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "Stop",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::Stop" );

  // Using API interface instead of FPGA in order to maintain consistancy with Run

  /*if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->RequesterStop( portHandle );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {*/
    RegWrite( portHandle, PCIE_STOP_ADDR, 32, 0x0 );
  /*}*/
}

void CAgtPCIEExerciser::RunFunctions
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtValueT functions
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RunFunctions",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtValueT, functions );

  TRACE( "CAgtPCIEExerciser::RunFunctions" );
 
  RegWrite( portHandle, PCIE_RUN_FUNCTIONS_ADDR, 32, functions );
}
        
void CAgtPCIEExerciser::StopFunctions
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtValueT functions
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "StopFunctions",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtValueT, functions );

  TRACE( "CAgtPCIEExerciser::StopFunctions" );

  RegWrite( portHandle, PCIE_STOP_FUNCTIONS_ADDR, 32, functions );
}

#pragma endregion

#pragma region Send Immediate

void CAgtPCIEExerciser::SiReqMemDWSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT dwNum,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqMemDWSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, dwNum,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::SiReqMemDWSet" );

  if( dwNum < PCIE_SI_REQ_PYLD_DW_SIZE )
  {  
    if( myController->IsISFPGAEnabled( portHandle ) )
    {
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      mSiMgmt->SiReqPyldSet( portHandle, ( UInt8 )dwNum, val );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
    }
    else
    {
      return RegWrite( portHandle, PCIE_SI_REQ_PYLD_DW_START_ADDR + dwNum, 32, val );
    }
  }
  else
  {
    AGT_THROW( "SiReqMemDWSet: Parameter dwNum must be less than PCIE_SI_REQ_PYLD_DW_SIZE" );
  }
}

void CAgtPCIEExerciser::SiReqSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIESi prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIESi, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::SiReqSet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    mSiMgmt->SiReqPropWrite( portHandle, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {  
    return RegWrite( portHandle, PCIE_SI_REQ_PROP_ADDR + ( UInt32 )prop, 32, val );
  }
}

void CAgtPCIEExerciser::SiReqGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIESi prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIESi, prop );

  TRACE( "CAgtPCIEExerciser::SiReqGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = mSiMgmt->SiReqPropRead( portHandle, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  { 
    return RegRead( portHandle, PCIE_SI_REQ_PROP_ADDR + ( AgtValueT )prop, 32, val );
  }
}

void CAgtPCIEExerciser::SiReqMemDWGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT dwNum,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqMemDWGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, dwNum );

  TRACE( "CAgtPCIEExerciser::SiReqMemDWGet" );

  if( dwNum < PCIE_SI_REQ_PYLD_DW_SIZE )
  {
    if( myController->IsISFPGAEnabled( portHandle ) )
    {
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = mSiMgmt->SiReqPyldGet( portHandle, ( UInt8 )dwNum );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
    }
    else
    {
      return RegRead( portHandle, PCIE_SI_REQ_PYLD_DW_START_ADDR + dwNum, 32, val );
    }
  }
  else
  {
    AGT_THROW( "SiReqMemDWGet: Parameter dwNum must be less than PCIE_SI_REQ_PYLD_DW_SIZE." );
  }
}

void CAgtPCIEExerciser::SiReqMemGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT numBytes,
  /* [out][in] */ AgtSizeT *pCount,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqMemGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, numBytes );

  TRACE( "CAgtPCIEExerciser::SiReqMemGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    mSiMgmt->SiReqPyldByteGet( portHandle, numBytes, psaData );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );

    *pCount = numBytes;
  }
  else
  {
    return MemRead( portHandle, PCIE_SI_REQ_PYLD_ADDR, numBytes, 8, pCount, psaData );
  }
}

void CAgtPCIEExerciser::SiReqMemSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT numBytes,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiReqMemSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, numBytes );

  TRACE( "CAgtPCIEExerciser::SiReqMemSet" );

  if( ( numBytes % 4 ) || ( numBytes == 0 ) || ( numBytes > 16 ) )
  {
    AGT_THROW( "SiReqMemSet: Parameter numBytes must equal 4, 8, 12 or 16." );
  }

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    mSiMgmt->SiReqPyldByteSet( portHandle, numBytes, psaData );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return MemWrite( portHandle, PCIE_SI_REQ_PYLD_ADDR, numBytes, 8, psaData );
  }
}

void CAgtPCIEExerciser::SiCompGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIESi prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiCompGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIESi, prop );

  TRACE( "CAgtPCIEExerciser::SiCompGet" );

  if( prop == PCIE_SI_AUTOTAG || prop == PCIE_SI_RESOURCE ||
      prop == PCIE_SI_INTADDR || prop == PCIE_SI_BUSERROR )
  {
    // This makes no sense for completion packet. 
    AGT_THROW( "SiCompGet: Parameter prop has invalid value." );
  }

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = mSiMgmt->SiCompPropRead( portHandle, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return RegRead( portHandle, PCIE_SI_COMP_PROP_ADDR + ( UInt32 )prop, 32, val );
  }
}

void CAgtPCIEExerciser::SiCompMemDWGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT dwNum,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiCompMemDWGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, dwNum );

  TRACE( "CAgtPCIEExerciser::SiCompMemDWGet" );

  if( dwNum < PCIE_SI_COMP_PYLD_DW_SIZE )
  {
    if( myController->IsISFPGAEnabled( portHandle ) )
    {
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      *val = mSiMgmt->SiCompPyldGet( portHandle, ( UInt8 )dwNum );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
    }
    else
    {
      return RegRead( portHandle, PCIE_SI_COMP_PYLD_DW_START_ADDR + dwNum, 32, val );
    }
  }
  else
  {
    AGT_THROW( "SiCompMemDWGet: Parameter dwNum must be less than PCIE_SI_COMP_PYLD_DW_SIZE." );
  }
}

void CAgtPCIEExerciser::SiCompMemGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT numBytes,
  /* [out][in] */ AgtSizeT *pCount,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiCompMemGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, numBytes );

  TRACE( "CAgtPCIEExerciser::SiCompMemGet" );

  if( ( numBytes % 4 ) || ( numBytes == 0 ) || ( numBytes > 16 ) )
  {
    AGT_THROW( "SiCompMemGet: Parameter numBytes must equal 4, 8, 12 or 16." );
  }

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    mSiMgmt->SiCompPyldByteGet( portHandle, numBytes, psaData );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );

    *pCount = numBytes;
  }
  else
  {
    return MemRead( portHandle, PCIE_SI_COMP_PYLD_ADDR, numBytes, 8, pCount, psaData );
  }
}

void CAgtPCIEExerciser::SiSend
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiSend",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::SiSend" );

  return RegWrite( portHandle, PCIE_SI_SEND_ADDR, 32, 1 );
}

void CAgtPCIEExerciser::SiDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiDefaultSet",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::SiDefaultSet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    mSiMgmt->SiDefaultSet( portHandle );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return RegWrite( portHandle, PCIE_SI_DEFAULT_ADDR, 32, 1 );
  }
}

void CAgtPCIEExerciser::SiStatusGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIESiStatus status,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SiStatusGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIESiStatus, status );

  portHandle; // unsused: for compiler's sake

  TRACE( "CAgtPCIEExerciser::SiStatusGet" );

  return RegRead( portHandle, PCIE_SISTATUS_ADDR + ( UInt32 )status, 32, val );
}

#pragma endregion

#pragma region Block memory

#pragma region BlockGen related

void CAgtPCIEExerciser::BlockGenDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockGenDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel );

  TRACE( "CAgtPCIEExerciser::BlockGenDefaultSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->BlockGenDefaultWrite( portHandle, channel );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return RegWrite( portHandle, PCIE_BLOCK_GEN_DEFAULT_ADDR, 32, ( UInt8 )channel );
  }
}

void CAgtPCIEExerciser::BlockGenSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ EPCIEBlockGen prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockGenSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_EPCIEBlockGen, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::BlockGenSet" );

  BLOCKMEMORYSTOPCHECK

  // This should not be done using FPGA as some settings like function enabled, mode are only written
  // when run is called. May cause inconsistant behavior when using multiple functions.

  /*if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->BlockGenWrite( portHandle, channel, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {*/
    UInt32 address;

    switch( channel )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELA_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELB_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELC_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELD_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELE_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELBVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELBVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELCVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELCVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELDVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELDVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELEVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELEVF2_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "BlockGenSet: Parameter channel has invalid value." );
    }

    return RegWrite( portHandle, address, 32, val );
  /*}*/
}

void CAgtPCIEExerciser::BlockGenGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ EPCIEBlockGen prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockGenGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_EPCIEBlockGen, prop );

  TRACE( "CAgtPCIEExerciser::BlockGenGet" );

  // This should not be done using FPGA as some settings like function enabled, mode are only written
  // when run is called. May cause inconsistant behavior when using multiple functions.

  /*if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->BlockGenRead( portHandle, channel, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {*/
    UInt32 address;

    switch( channel )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELA_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELB_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELC_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELD_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELE_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELBVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELBVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELCVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELCVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELDVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELDVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELEVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      address = PCIE_EXERCISER_BLOCK_HWCHANNELEVF2_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "BlockGenGet: Parameter channel has invalid value." );
    }

    return RegRead( portHandle, address, 32, val );
  /*}*/
}

#pragma endregion

#pragma region Block line related

void CAgtPCIEExerciser::BlockDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line );

  TRACE( "CAgtPCIEExerciser::BlockDefaultSet" );

  BLOCKMEMORYSTOPCHECK;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->BlockLineDefaultSet( portHandle, channel, line );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    UInt32 theVal = ( UInt32 )line;

    theVal |= ( ( ( UInt32 )channel ) << PCIE_BLOCKLINE_CHANNEL_SEPARATOR_BIT );

    return RegWrite( portHandle, PCIE_BLOCKLINEDEFAULTSET_ADDR, 32, theVal );
  }
}

void CAgtPCIEExerciser::BlockSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIEBlock prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line,
    ParamType_EPCIEBlock, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::BlockSet" );

  BLOCKMEMORYSTOPCHECK;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->BlockSet( portHandle, channel, line, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetBlockPropAddress( channel, line, &address );
    RegWrite( portHandle, address, ( UInt8 ) prop, val );
  }
}

void CAgtPCIEExerciser::BlockGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIEBlock prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "BlockGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line,
    ParamType_EPCIEBlock, prop );

  TRACE( "CAgtPCIEExerciser::BlockGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->BlockGet( portHandle, channel, line, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetBlockPropAddress( channel, line, &address );
    RegRead( portHandle, address, ( UInt8 )prop, val );
  }
}

#pragma endregion

#pragma region ReqBehGen related

void CAgtPCIEExerciser::ReqBehGenSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ EPCIEReqBehGen prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehGenSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_EPCIEReqBehGen, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ReqBehGenSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->ReqBehGenWrite( portHandle, channel, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    UInt32 address;

    switch( channel )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELA_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELB_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELC_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELD_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELE_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELBVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELBVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELCVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELCVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELDVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELDVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELEVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELEVF2_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "ReqBehGenSet: Parameter channel has invalid value." );  
    }

    return RegWrite( portHandle, address, 32, val );
  }
}

void CAgtPCIEExerciser::ReqBehGenGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ EPCIEReqBehGen prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehGenGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_EPCIEReqBehGen, prop );

  TRACE( "CAgtPCIEExerciser::ReqBehGenGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->ReqBehGenRead( portHandle, channel, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    UInt32 address;

    switch( channel )
    {
    case PCIE_HWCHANNEL_FUNCTION_A:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELA_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_B:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELB_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_C:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELC_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_D:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELD_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_E:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELE_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELBVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_BVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELBVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELCVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_CVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELCVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELDVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_DVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELDVF2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF1:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELEVF1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_HWCHANNEL_FUNCTION_EVF2:
      address = PCIE_EXERCISER_REQBEH_HWCHANNELEVF2_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "ReqBehGenGet: Parameter channel has invalid value" );
    }

    return RegRead( portHandle, address, 32, val );
  }
}

void CAgtPCIEExerciser::ReqBehGenDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehGenDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel );

  TRACE( "CAgtPCIEExerciser::ReqBehGenDefaultSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->ReqBehGenDefaultWrite( portHandle, channel );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return RegWrite( portHandle, PCIE_REQBEH_GEN_DEFAULT_ADDR, 32, ( UInt8 )channel );
  }
}

#pragma endregion

#pragma region Behavior line related

void CAgtPCIEExerciser::ReqBehSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIEReqBeh prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line,
    ParamType_EPCIEReqBeh, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ReqBehSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->ReqBehSet( portHandle, channel, line, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetReqBehPropAddress( channel, line, &address );

    return RegWrite( portHandle, address, ( UInt8 )prop, val );
  }
}

void CAgtPCIEExerciser::ReqBehGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIEReqBeh prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line,
    ParamType_EPCIEReqBeh, prop );

  TRACE( "CAgtPCIEExerciser::ReqBehGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->ReqBehGet( portHandle, channel, line, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetReqBehPropAddress( channel, line, &address );

    return RegRead( portHandle, address, ( UInt8 )prop, val );
  }
}

void CAgtPCIEExerciser::ReqBehDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction channel,
  /* [in] */ AgtSizeT line
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ReqBehDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, channel,
    ParamType_AgtSizeT, line );

  TRACE( "CAgtPCIEExerciser::ReqBehDefaultSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->ReqBehLineDefaultSet( portHandle, channel, line );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    UInt32 theVal = ( UInt32 )line;

    theVal |= ( ( ( UInt32 )channel ) << PCIE_REQBEHLINE_CHANNEL_SEPARATOR_BIT );

    return RegWrite( portHandle, PCIE_REQBEHLINEDEFAULTSET_ADDR, 32, theVal );
  }
}

#pragma endregion

#pragma endregion

#pragma region Completion behavior

void CAgtPCIEExerciser::CompBehSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIECompBeh prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue,
    ParamType_AgtSizeT, line,
    ParamType_EPCIECompBeh, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::CompBehSet" );

  BLOCKMEMORYSTOPCHECK;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->CompBehSet( portHandle, queue, line, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetCompBehPropAddress( queue, line, &address );
    return RegWrite( portHandle, address, ( UInt8 )prop, val );
  }
}

void CAgtPCIEExerciser::CompBehGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue,
  /* [in] */ AgtSizeT line,
  /* [in] */ EPCIECompBeh prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue,
    ParamType_AgtSizeT, line,
    ParamType_EPCIECompBeh, prop );

  TRACE( "CAgtPCIEExerciser::CompBehGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->CompBehGet( portHandle, queue, line, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    AgtValueT address = 0;

    GetCompBehPropAddress( queue, line, &address );
    return RegRead( portHandle, address, ( UInt8 )prop, val );
  }
}

void CAgtPCIEExerciser::CompBehDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue,
  /* [in] */ AgtSizeT line
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue,
    ParamType_AgtSizeT, line );

  TRACE( "CAgtPCIEExerciser::CompBehDefaultSet" );

  BLOCKMEMORYSTOPCHECK;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->CompBehLineDefaultSet( portHandle, queue, line );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    UInt32 theVal = ( UInt32 )line;

    theVal |= ( ( ( UInt32 )queue ) << PCIE_COMPBEHLINE_CHANNEL_SEPARATOR_BIT );

    return RegWrite( portHandle, PCIE_COMPBEHLINEDEFAULTSET_ADDR, 32, theVal );
  }
}

void CAgtPCIEExerciser::CompBehGenSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue,
  /* [in] */ EPCIECompBehGen prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehGenSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue,
    ParamType_EPCIECompBehGen, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::CompBehGenSet" );

  BLOCKMEMORYSTOPCHECK;

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->CompBehGenWrite( portHandle, queue, prop, val );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {  
    UInt32 address;

    switch( queue )
    {
    case PCIE_COMPQUEUE_0:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_0_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_1:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_2:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_3:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_3_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_4:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_4_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "CompBehGenSet: Parameter queue has invalid value." );
    }

    return RegWrite( portHandle, address, 32, val );
  }
}

void CAgtPCIEExerciser::CompBehGenGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue,
  /* [in] */ EPCIECompBehGen prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehGenGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue,
    ParamType_EPCIECompBehGen, prop );

  TRACE( "CAgtPCIEExerciser::CompBehGenGet" );

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    *val = myMemoryMgmt->CompBehGenRead( portHandle, queue, prop );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {  
    UInt32 address;

    switch( queue )
    {
    case PCIE_COMPQUEUE_0:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_0_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_1:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_1_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_2:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_2_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_3:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_3_ADDR + ( UInt32 )prop;
      break;
    case PCIE_COMPQUEUE_4:
      address = PCIE_EXERCISER_COMPBEH_QUEUE_4_ADDR + ( UInt32 )prop;
      break;
    default:
      AGT_THROW( "CompBehGenGet: Parameter queue has invalid value." );
    }

    return RegRead( portHandle, address, 32, val );
  }
}

void CAgtPCIEExerciser::CompBehGenDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIECompQueue queue
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "CompBehGenDefaultSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIECompQueue, queue );

  TRACE( "CAgtPCIEExerciser::CompBehGenDefaultSet" );

  BLOCKMEMORYSTOPCHECK

  if( myController->IsISFPGAEnabled( portHandle ) )
  {
    myController->GetFunctionLockAndFPGAAccess( portHandle );
    myMemoryMgmt->CompBehGenDefaultWrite( portHandle, queue );
    myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
  }
  else
  {
    return RegWrite( portHandle, PCIE_COMPBEH_GEN_DEFAULT_ADDR, 32, ( UInt8 )queue );
  }
}

#pragma endregion

#pragma endregion

#pragma region Data memory related

void CAgtPCIEExerciser::DataMemRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtSizeT length,
  /* [out][in] */ AgtSizeT *pCount,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DataMemRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset,
    ParamType_AgtSizeT, length );

  TRACE( "CAgtPCIEExerciser::DataMemRead" );

  BLOCKMEMORYSTOPCHECK

  if( offset + length <= PCIE_DATAMEM_SIZE )
  {
    if( myController->IsISFPGAEnabled( portHandle ) )
    {
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      myFpgaFramework->getFpgaCSDat().readDataHotAccess( portHandle, offset, length, psaData, pCount );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
    }
    else
    {
      MemRead( portHandle, PCIE_DATAMEMSTART_ADDR + offset, length, 32, pCount, psaData );
    }
  }
  else
  {
    AGT_THROW( "DataMemRead: Parameters offset+length must be less equal PCIE_DATAMEM_SIZE." );
  }
}

void CAgtPCIEExerciser::DataMemWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtSizeT length,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DataMemWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset,
    ParamType_AgtSizeT, length );

  TRACE( "CAgtPCIEExerciser::DataMemWrite" );

  BLOCKMEMORYSTOPCHECK

  if( offset + length <= PCIE_DATAMEM_SIZE )
  {
    if( myController->IsISFPGAEnabled( portHandle ) )
    {
      myController->GetFunctionLockAndFPGAAccess( portHandle );
      myFpgaFramework->getFpgaCSDat().writeDataHotAccess( portHandle, offset, length, psaData );
      myController->ReleaseFunctionLockAndFPGAAccess( portHandle, mReleaseFPGAAfterEachCall );
    }
    else
    {
      MemWrite( portHandle, PCIE_DATAMEMSTART_ADDR + offset, length, 32, psaData );
    }
  }
  else
  {
    AGT_THROW( "DataMemWrite: Parameters offset+length must be less equal PCIE_DATAMEM_SIZE." );
  }
}

#pragma endregion

#pragma region Config space related

#pragma region Direct read, write

// Not supported, not documented.
// Executes a direct config access without using
// the PCI API protocol.
void CAgtPCIEExerciser::ConfRegDirectRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegDirectRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset );

  TRACE( "CAgtPCIEExerciser::ConfRegDirectRead" );

  AgtValueT value;

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW( "ConfRegDirectRead: Invalid parameter offset. Must be DW-aligned" );
  }

  myController->ConfRegDirectReadWithFnLock( portHandle, offset, value );

  *val = value;
}

// Not supported, not documented.
// Executes a direct config access without using
// the PCI API protocol.
void CAgtPCIEExerciser::ConfRegDirectWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegDirectWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ConfRegDirectWrite" );

  BLOCKMEMORYSTOPCHECK

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW( "ConfRegDirectWrite: Invalid parameter offset. Must be DW-aligned" );
  }

  return myController->ConfRegDirectWriteWithFnLock( portHandle, offset, val );
}

#pragma endregion

#pragma region ConfReg read, write

void CAgtPCIEExerciser::ConfRegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT offset,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, offset );

  TRACE( "CAgtPCIEExerciser::ConfRegRead" );

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW( "ConfRegRead: Invalid parameter offset. Must be DW-aligned" );
  }

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfRegRead: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  if( ( offset >= PCIE_CFGPORT_DWSTART * 4 && offset <= PCIE_CFGPORT_DWEND * 4 ) &&
      ( offset != PCIE_MODULENUMBER_FOR_IBM * 4 ) )
  {
    // We do not allow accessing the CFG port registers via the PCIE port API protocol,
    // because this would lead to an infinite recursion in the ESW and/or
    // disturb the port PCIE API SM (see AgtOSPort32.cpp).
    if( val )
    {
      *val = 0xaffebaff; // indicate nonsense
    }
  }
  else
  {
    UInt32 uintVal = 0;

    myController->SetConfSpaceFuncNumAndRegRead( portHandle, functionNum, offset + PCIE_CONFIGSPACE_ADDR, 32, uintVal );

    *val = uintVal;
  }
}

void CAgtPCIEExerciser::ConfRegWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, offset,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ConfRegWrite" );

  BLOCKMEMORYSTOPCHECK

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW( "ConfRegWrite: Invalid parameter offset. Must be DW-aligned" );
  }

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfRegWrite: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  if( offset >= PCIE_CFGPORT_DWSTART * 4 && offset <= PCIE_CFGPORT_DWEND * 4 )
  {
    // We do not allow accessing the CFG port registers via the PCIE port API protocol,
    // because this would lead to an infinite recursion in the ESW and/or
    // disturb the port PCIE API SM (see AgtOSPort32.cpp).

    // Do nothing.
  }
  else
  {
    myController->SetConfSpaceFuncNumAndRegWrite( portHandle, functionNum, offset + PCIE_CONFIGSPACE_ADDR, 32, val );
  }
}

void CAgtPCIEExerciser::ConfRegMaskRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT offset,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegMaskRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, offset );

  TRACE( "CAgtPCIEExerciser::ConfRegMaskRead" );

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW("ConfRegMaskRead: Invalid parameter offset. Must be DW-aligned" );
  }

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfRegMaskRead: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  UInt32 uintVal = 0;

  myController->SetConfSpaceFuncNumAndRegRead( portHandle, functionNum, offset + PCIE_CONFIGSPACE_MASK_ADDR, 32, uintVal );

  *val = uintVal;
}

void CAgtPCIEExerciser::ConfRegMaskWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfRegMaskWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, offset,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ConfRegMaskWrite" );

  if( CONFREGOFFSETCHECK( offset ) )
  {
    AGT_THROW( "ConfRegMaskWrite: Invalid parameter offset. Must be DW-aligned." );
  }

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfRegMaskWrite: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  return myController->SetConfSpaceFuncNumAndRegWrite( portHandle, functionNum, offset + PCIE_CONFIGSPACE_MASK_ADDR, 32, val );
}

#pragma endregion

#pragma region ConfigSpace read, write

void CAgtPCIEExerciser::ConfigSpaceRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEConfigSpace prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfigSpaceRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEConfigSpace, prop );

  TRACE( "CAgtPCIEExerciser::ConfigSpaceRead" );

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfigSpaceRead: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  UInt32 uintVal = 0;

  myController->SetConfSpaceFuncNumAndRegRead( portHandle, functionNum, PCIE_CONFIGSPACEPROP_ADDR + ( UInt32 )prop, 32, uintVal );

  *val = uintVal;
}

void CAgtPCIEExerciser::ConfigSpaceWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEConfigSpace prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ConfigSpaceWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEConfigSpace, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::ConfigSpaceWrite" );

  BLOCKMEMORYSTOPCHECK

  if( functionNum > PCIE_HWCHANNEL_FUNCTION_E )
    AGT_THROW( "ConfigSpaceWrite: Invalid parameter 'functionNum'. 'functionNum' should be one of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );

  myController->SetConfSpaceFuncNumAndRegWrite( portHandle, functionNum, PCIE_CONFIGSPACEPROP_ADDR + ( UInt32 )prop, 32, val );
}

#pragma endregion

#pragma region Function Table

void CAgtPCIEExerciser::FTRegRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "FTRegRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset );

  TRACE( "CAgtPCIEExerciser::FTRegRead" );

  RegRead( portHandle, offset + PCIE_FT_ADDR, 32, val );
}

void CAgtPCIEExerciser::FTRegWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT offset,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "FTRegWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, offset,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::FTRegWrite" );

  RegWrite( portHandle, offset + PCIE_FT_ADDR, 32, val );
}

#pragma endregion

#pragma endregion

#pragma region Decoder related

void CAgtPCIEExerciser::DecWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEDec dec,
  /* [in] */ EPCIEDecProp prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DecWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEDec, dec,
    ParamType_EPCIEDecProp, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::DecWrite" );

  BLOCKMEMORYSTOPCHECK

  UInt32 address = getDecTypePropertyAddress( dec, prop );

  switch( functionNum )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    address += PCIE_ADDR_DECFUNC_A;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    address += PCIE_ADDR_DECFUNC_B;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    address += PCIE_ADDR_DECFUNC_C;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    address += PCIE_ADDR_DECFUNC_D;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    address += PCIE_ADDR_DECFUNC_E;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    address += PCIE_ADDR_DECFUNC_BVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    address += PCIE_ADDR_DECFUNC_BVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    address += PCIE_ADDR_DECFUNC_CVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    address += PCIE_ADDR_DECFUNC_CVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    address += PCIE_ADDR_DECFUNC_DVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    address += PCIE_ADDR_DECFUNC_DVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    address += PCIE_ADDR_DECFUNC_EVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    address += PCIE_ADDR_DECFUNC_EVF2;
    break;
  default:
    AGT_THROW( "DecWrite: Parameter functionNum is invalid" );
  }

  return RegWrite( portHandle, address, 32, val );
}

void CAgtPCIEExerciser::DecRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEDec dec,
  /* [in] */ EPCIEDecProp prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DecRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEDec, dec,
    ParamType_EPCIEDecProp, prop );

  TRACE( "CAgtPCIEExerciser::DecRead" );

  UInt32 address = getDecTypePropertyAddress( dec, prop );

  switch( functionNum )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    address += PCIE_ADDR_DECFUNC_A;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    address += PCIE_ADDR_DECFUNC_B;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    address += PCIE_ADDR_DECFUNC_C;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    address += PCIE_ADDR_DECFUNC_D;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    address += PCIE_ADDR_DECFUNC_E;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF1:
    address += PCIE_ADDR_DECFUNC_BVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_BVF2:
    address += PCIE_ADDR_DECFUNC_BVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF1:
    address += PCIE_ADDR_DECFUNC_CVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_CVF2:
    address += PCIE_ADDR_DECFUNC_CVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF1:
    address += PCIE_ADDR_DECFUNC_DVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_DVF2:
    address += PCIE_ADDR_DECFUNC_DVF2;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF1:
    address += PCIE_ADDR_DECFUNC_EVF1;
    break;
  case PCIE_HWCHANNEL_FUNCTION_EVF2:
    address += PCIE_ADDR_DECFUNC_EVF2;
    break;
  default:
    AGT_THROW( "DecRead: Parameter functionNum is invalid" );
  }

  return RegRead( portHandle, address, 32, val );
}

#pragma endregion

#pragma region Protocol Error

#pragma region General

void CAgtPCIEExerciser::ProtocolErrorStatusClear
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorStatusClear",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorStatusClear" );

  // Clears all accu and first errors
  AgtValueT dummy = 0;

  return RegWrite( portHandle, PCIE_PROTRULE_CLEAR_ADDR, 32, dummy );
}

void CAgtPCIEExerciser::ProtocolErrorStatusDump
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ char **errortext
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorStatusDump",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorStatusDump" );

  // Returns a textual status of the protocol checker
  AgtValueT accuErr = 0;
  AgtValueT firstErr = 0;
  AgtValueT mask = 0;
  static char statusString[0x10000];
  char *ruleDescription = NULL;
  int rule = 0;
  bool bErrorFound = false;

  statusString[0] = '\0';
  strcat( statusString, "Errors found by protocol checker:\n" );

  const int nNumRules = 12;

  EPCIEProtRule rules[nNumRules];

  rules[0] = PCIE_PROTRULE_UNSUPPORTED_REQUEST;
  rules[1] = PCIE_PROTRULE_DECODER_MISS;
  rules[2] = PCIE_PROTRULE_NULLIFIED_LCRC;
  rules[3] = PCIE_PROTRULE_BAD_EDB_LCRC;
  rules[4] = PCIE_PROTRULE_NAK_SENT;
  rules[5] = PCIE_PROTRULE_TLP_SEQUENCE;
  rules[6] = PCIE_PROTRULE_DLLP_CRC;
  rules[7] = PCIE_PROTRULE_DESKEW_ALIGN;
  rules[8] = PCIE_PROTRULE_DISPARITY;
  rules[9] = PCIE_PROTRULE_CODING;
  rules[10] = PCIE_PROTRULE_128B_130B_SYNC_ERROR;
  rules[11] = PCIE_PROTRULE_PCIE_ECRC_ERROR;

  // Physical layer errors
  for( int i = 0; i < nNumRules; i++ )
  {
    rule = rules[i];

    // Read status of rule
    ProtocolErrorAccuRead( portHandle, ( EPCIEProtRule )rule, &accuErr );

    if( accuErr )
    {
      bErrorFound = true;

      // Error occurred, add rule description to statusString
      ProtocolErrorStringGet( portHandle, ( EPCIEProtRule )rule, &ruleDescription );
      strcat( statusString, ruleDescription );

      // Read additional info
      ProtocolErrorFirstRead( portHandle, ( EPCIEProtRule )rule, &firstErr );
      if( firstErr )
      {
        strcat( statusString, " [first Error]" );
      }

      ProtocolErrorMaskGet( portHandle, ( EPCIEProtRule )rule, &mask );
      if( mask )
      {
        strcat( statusString, " [masked]" );
      }

      strcat( statusString, "\n" );
    }
  }

  for( int channel = 0; channel < PCIE_NUM_VCRESOURCES; channel++ )
  {
    for( rule = 0; rule < PCIE_PROTRULE_VC_END; rule++ )
    {
      // Read status of rule
      ProtocolErrorVCAccuRead( portHandle, ( EPCIEVCResourceId )channel, ( EPCIEProtRuleVC )rule, &accuErr );

      if( accuErr )
      {
        bErrorFound = true;

        // Error occurred, add rule description to statusString
        ProtocolErrorVCStringGet( portHandle, ( EPCIEProtRuleVC )rule, &ruleDescription );
        strcat( statusString, ruleDescription );

        // Read additional info
        ProtocolErrorVCFirstRead( portHandle, ( EPCIEVCResourceId )channel, ( EPCIEProtRuleVC )rule, &firstErr );
        if( firstErr )
        {
          strcat( statusString, " [first Error]" );
        }

        ProtocolErrorVCMaskGet( portHandle, ( EPCIEVCResourceId )channel, ( EPCIEProtRuleVC )rule, &mask );
        if( mask == 0 )
        {
          strcat( statusString, " [masked]" );
        }

        strcat( statusString, "\n" );
      }
    }
  }

  for( int funcNum = 0; funcNum < PCIE_NUM_FUNCTIONS; funcNum++ )
  {
    for( rule = 0; rule < PCIE_PROTRULE_FUNCTION_END; rule++ )
    {
      // Read status of rule
      ProtocolErrorFunctionAccuRead( portHandle, ( EPCIEHwChannelFunction )funcNum, ( EPCIEProtRuleFunction )rule, &accuErr );

      if( accuErr )
      {
        bErrorFound = true;

        // Error occurred, add rule description to statusString
        ProtocolErrorFunctionStringGet( portHandle, ( EPCIEProtRuleFunction )rule, &ruleDescription );
        strcat( statusString, ruleDescription );

        // Read additional info
        ProtocolErrorFunctionFirstRead( portHandle, ( EPCIEHwChannelFunction )funcNum, ( EPCIEProtRuleFunction )rule, &firstErr );
        if( firstErr )
        {
          strcat( statusString, " [first Error]" );
        }

        ProtocolErrorFunctionMaskGet( portHandle, ( EPCIEHwChannelFunction )funcNum, ( EPCIEProtRuleFunction )rule, &mask );
        if( mask == 0 )
        {
          strcat( statusString, " [masked]" );
        }

        strcat( statusString, "\n" );
      }
    }
  }

  if( !bErrorFound )
  {
    // No error at all
    strcat( statusString, "None.\n" );
  }

  *errortext = statusString;
}

#pragma endregion

#pragma region Common

void CAgtPCIEExerciser::ProtocolErrorAccuRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRule rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorAccuRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRule, rule );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorAccuRead" );

  // Read status of accu error for protocol rule
  return RegRead( portHandle, PCIE_PROTRULE_ACCU_ADDR + ( AgtValueT )rule, 32, val );
}

void CAgtPCIEExerciser::ProtocolErrorFirstRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRule rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFirstRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRule, rule );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorFirstRead" );

  // Read status of first error for protocol rule
  return RegRead( portHandle, PCIE_PROTRULE_FIRST_ADDR + ( AgtValueT )rule, 32, val );
}

void CAgtPCIEExerciser::ProtocolErrorMaskSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRule rule,
  /* [in] */ AgtValueT mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorMaskSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRule, rule,
    ParamType_AgtValueT, mask );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorMaskSet" );

  // Set error mask for protocol rule
  return RegWrite( portHandle, PCIE_PROTRULE_MASK_ADDR + ( AgtValueT )rule, 32, mask );
}

void CAgtPCIEExerciser::ProtocolErrorMaskGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRule rule,
  /* [retval][out] */ AgtValueT *mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorMaskGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRule, rule );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorMaskGet" );

  // Get error mask for protocol rule
  return RegRead( portHandle, PCIE_PROTRULE_MASK_ADDR + ( AgtValueT )rule, 32, mask );
}

void CAgtPCIEExerciser::ProtocolErrorStringGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRule rule,
  /* [retval][out] */ char **errortext
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorStringGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRule, rule );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorStringGet" );

  // Returns a textual description of the given protocol rule.

  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

  \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRule)
  \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
  \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)

  /****************************************************************************************************/

  portHandle; // unused parameter

  switch( rule )
  {
  case PCIE_PROTRULE_UNSUPPORTED_REQUEST:
    *errortext = "PCIE_PROTRULE_UNSUPPORTED_REQUEST: The type of an incoming request is not supported. An unsupported request completion is generated (currently only for MsgASD). See PCI Express 1.0a, section 2.3.1, \"If the Request Type is not supported by the device, the Request is an Unsupported Request and is reported ...\"";
    break;
  case PCIE_PROTRULE_DECODER_MISS:
    *errortext = "PCIE_PROTRULE_DECODER_MISS: Received request packet not covered by any decoder. If subtractive decode is enabled no decoder miss can occur. In this case decoder misses are caught by the subtractive decoder.";
    break;
  case PCIE_PROTRULE_NULLIFIED_LCRC:
    *errortext = "PCIE_PROTRULE_NULLIFIED_LCRC: Received packet with nullified LCRC";
    break;
  case PCIE_PROTRULE_BAD_EDB_LCRC:
    *errortext = "PCIE_PROTRULE_BAD_EDB_LCRC: TLP with bad end EDB or bad (not poisoned) LCRC";
    break;
  case PCIE_PROTRULE_NAK_SENT:
    *errortext = "PCIE_PROTRULE_NAK_SENT: NAK sent";
    break;
  case PCIE_PROTRULE_DLLP_CRC:
    *errortext = "PCIE_PROTRULE_DLLP_CRC: DLLP checksum error";
    break;
  case PCIE_PROTRULE_TLP_SEQUENCE:
    *errortext = "PCIE_PROTRULE_TLP_SEQUENCE: Received an out-of-sequence TLP. The protocol check finds two different incorrect behaviors: - The exerciser receives an ACK or NAK with sequence number A. After that, it receives an ACK or NAK with a predecessor of A. A sequence number X is a predecessor of A if [(X-A) MOD 2^12 > 2^11]. - The exerciser receives an ACK or NAK with sequence number A and the TLP with sequence number A has not yet been sent (or has been sent far in the past). If X is the sequence number of the last transmitted TLP then A is an incorrect sequence number if [(X-A) MOD 2^12 > 2^11].";
    break;
  case PCIE_PROTRULE_DESKEW_ALIGN:
    *errortext = "PCIE_PROTRULE_DESKEW_ALIGN: Deskew alignment error. Lanes fell out of alignment";
    break;
  case PCIE_PROTRULE_DISPARITY:
    *errortext = "PCIE_PROTRULE_DISPARITY: Running disparity error";
    break;
  case PCIE_PROTRULE_CODING:
    *errortext = "PCIE_PROTRULE_CODING: Coding error";
    break;
  case PCIE_PROTRULE_128B_130B_SYNC_ERROR:
    *errortext = "PCIE_PROTRULE_128B_130B_SYNC_ERROR: 128b/130b Sync error";
    break;
  case PCIE_PROTRULE_PCIE_ECRC_ERROR:
    *errortext = "PCIE_PROTRULE_PCIE_ECRC_ERROR: ECRC error detected";
    break;
  default:
    *errortext = "ERROR (invalid parameter): Unknown rule passed to CAgtPCIEExerciser::ProtocolErrorStringGet";
    break;
  }
}

#pragma endregion

#pragma region Function specific

void CAgtPCIEExerciser::ProtocolErrorFunctionAccuRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEProtRuleFunction rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFunctionAccuRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEProtRuleFunction, rule );

  TRACE( "CAgtPCIEExerciser::ProtocolErrorFunctionAccuRead" );

  // Read status of accu error for protocol rule
  return RegRead( portHandle, PCIE_PROTRULE_FUNCTION_ACCU_ADDR + ( AgtValueT )rule, functionNum, val );
}

void CAgtPCIEExerciser::ProtocolErrorFunctionFirstRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEProtRuleFunction rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFunctionFirstRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEProtRuleFunction, rule );

  // Read status of first error for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorFunctionFirstRead" );

  return RegRead( portHandle, PCIE_PROTRULE_FUNCTION_FIRST_ADDR + ( AgtValueT )rule, functionNum, val );
}

void CAgtPCIEExerciser::ProtocolErrorFunctionMaskSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEProtRuleFunction rule,
  /* [in] */ AgtValueT mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFunctionMaskSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEProtRuleFunction, rule,
    ParamType_AgtValueT, mask);

  // Set error mask for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorFunctionMaskSet" );

  return RegWrite( portHandle, PCIE_PROTRULE_FUNCTION_MASK_ADDR + ( AgtValueT )rule, functionNum, mask );
}

void CAgtPCIEExerciser::ProtocolErrorFunctionMaskGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEProtRuleFunction rule,
  /* [retval][out] */ AgtValueT *mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFunctionMaskGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEProtRuleFunction, rule );

  // Get error mask for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorFunctionMaskGet" );

  return RegRead( portHandle, PCIE_PROTRULE_FUNCTION_MASK_ADDR + ( AgtValueT )rule, functionNum, mask );
}

void CAgtPCIEExerciser::ProtocolErrorFunctionStringGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRuleFunction rule,
  /* [retval][out] */ char **errortext
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorFunctionStringGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRuleFunction, rule );

  // Returns a textual description of the given protocol rule.
  TRACE( "CAgtPCIEExerciser::ProtocolErrorFunctionStringGet" );

  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

  \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRuleFunction)
  \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
  \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)

  /****************************************************************************************************/

  portHandle; // unused parameter

  switch( rule )
  {
  case PCIE_PROTRULE_UNEXPECTED_COMPLETION:
    *errortext = "PCIE_PROTRULE_UNEXPECTED_COMPLETION: Received an unexpected completion packet, i.e. cannot associate with any outstanding request. The tag from the incoming completion is blocked after that. It is not used anymore by the exerciser until the tags are initialized (tag reset or link training).";
    break;
  case PCIE_PROTRULE_COMPLETION_TIMEOUT:
    *errortext = "PCIE_PROTRULE_COMPLETION_TIMEOUT: The exerciser has detected that it has not received an outstanding completion within a programmable time. It is not guaranteed that the exerciser detects a completion timeout immediately.";
    break;
  case PCIE_PROTRULE_COMPLETION_DATA:
    *errortext = "PCIE_PROTRULE_COMPLETION_DATA: The last completion for a request contains more data than expected. The tag from the incoming completion is blocked after that. It is not used anymore by the exerciser until the tags are initialized (tag reset or link training).";
    break;
  default:
    *errortext = "ERROR (invalid parameter): Unknown rule passed to CAgtPCIEExerciser::ProtocolErrorFunctionStringGet";
    break;
  }
}

#pragma endregion

#pragma region VC specific

void CAgtPCIEExerciser::ProtocolErrorVCAccuRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEProtRuleVC rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorVCAccuRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEProtRuleVC, rule );

  // Read status of accu error for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorVCAccuRead" );

  return RegRead( portHandle, PCIE_PROTRULE_VC_ACCU_ADDR + ( AgtValueT )rule, channel, val );
}

void CAgtPCIEExerciser::ProtocolErrorVCFirstRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEProtRuleVC rule,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorVCFirstRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEProtRuleVC, rule );

  // Read status of first error for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorVCFirstRead" );

  return RegRead( portHandle, PCIE_PROTRULE_VC_FIRST_ADDR + ( AgtValueT )rule, channel, val );
}

void CAgtPCIEExerciser::ProtocolErrorVCMaskSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEProtRuleVC rule,
  /* [in] */ AgtValueT mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorVCMaskSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEProtRuleVC, rule,
    ParamType_AgtValueT, mask );

  // Set error mask for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorVCMaskSet" );

  return RegWrite( portHandle, PCIE_PROTRULE_VC_MASK_ADDR + ( AgtValueT )rule, channel, mask );
}

void CAgtPCIEExerciser::ProtocolErrorVCMaskGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEProtRuleVC rule,
  /* [retval][out] */ AgtValueT *mask
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorVCMaskGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEProtRuleVC, rule );

  // Get error mask for protocol rule
  TRACE( "CAgtPCIEExerciser::ProtocolErrorVCMaskGet" );

  return RegRead( portHandle, PCIE_PROTRULE_VC_MASK_ADDR + ( AgtValueT )rule, channel, mask );
}

void CAgtPCIEExerciser::ProtocolErrorVCStringGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEProtRuleVC rule,
  /* [retval][out] */ char **errortext
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ProtocolErrorVCStringGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEProtRuleVC, rule );

  // Returns a textual description of the given protocol rule.
  TRACE( "CAgtPCIEExerciser::ProtocolErrorVCStringGet" );

  /****************************************************************************************************/
  /****** Please keep rule descriptions consistent within these files:

  \ipcommon\shared\dev\devpciedefs.h (docu in enum EPCIEProtRuleVC)
  \ipmodel\host\system\sessions\PCIExpress\Pciapi\AgtPCIEExerciser.cpp (in-system port API)
  \ipmodel\host\system\subsystems\pciexpress\AgtPCIEExerciser.cpp (Exerciser Interface implementation)

  /****************************************************************************************************/

  portHandle; // unused parameter

  switch( rule )
  {
  case PCIE_PROTRULE_FC_TIMEOUT:
    *errortext = "PCIE_PROTRULE_FC_TIMEOUT: Flow control update packet not received. PCI Express, section 2.6.1.2, sub-section \"Flow Control Update Frequency\"";
    break;
  case PCIE_PROTRULE_MRIOV_ECRC_ERROR:
    *errortext = "PCIE_PROTRULE_MRIOV_ECRC_ERROR: ECRC error detected";
    break;
  default:
    *errortext = "ERROR (invalid parameter): Unknown rule passed to CAgtPCIEExerciser::ProtocolErrorVCStringGet";
    break;
  }
}

#pragma endregion

#pragma endregion

#pragma region Status related functions

void CAgtPCIEExerciser::TxStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "TxStatusRead",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::TxStatusRead" );

  return RegRead( portHandle, PCIE_TXSTATUS_ADDR, 32, val );
}

void CAgtPCIEExerciser::RxStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxStatusRead",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::RxStatusRead" );

  return RegRead( portHandle, PCIE_RXSTATUS_ADDR, 32, val );
}

void CAgtPCIEExerciser::LinkWidthStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "LinkWidthStatusRead",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::LinkWidthStatusRead" );

  return RegRead( portHandle, PCIE_LINKWIDTHSTATUS_ADDR, 32, val );
}

void CAgtPCIEExerciser::DataLinkStateRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DataLinkStateRead",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::DataLinkStateRead" );

  return RegRead( portHandle, PCIE_DATALINKSTATE_ADDR, 32, val );
}

void CAgtPCIEExerciser::ExerciserStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEExerciserStatus status,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEExerciserStatus, status );

  TRACE( "CAgtPCIEExerciser::ExerciserStatusRead" );

  if( ExerciserStatusReadFunction( portHandle, status, val ) == false )
  {
    RegRead( portHandle, PCIE_EXERCISERSTATUS_ADDR + ( UInt32 )status, 32, val );
  }
}

void CAgtPCIEExerciser::ExerciserStatusReset
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserStatusReset",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::ExerciserStatusReset" );

  return RegWrite( portHandle, PCIE_EXERCISER_STATUS_RESET_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::ExerciserStatusSnapshot
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserStatusSnapshot",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::ExerciserStatusSnapshot" );

  return RegWrite( portHandle, PCIE_EXERCISER_STATUS_SNAPSHOT_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::ExerciserStatusFunctionReset
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserStatusFunctionReset",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum );

  TRACE( "CAgtPCIEExerciser::ExerciserStatusFunctionReset" );

  return RegWrite( portHandle, PCIE_EXERCISER_STATUS_RESET_FUNCTION_ADDR + ( int )functionNum, 32, 0 );
}

void CAgtPCIEExerciser::ExerciserStatusFunctionSnapshot
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserStatusFunctionSnapshot",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum );

  TRACE( "CAgtPCIEExerciser::ExerciserStatusFunctionSnapshot" );

  return RegWrite( portHandle, PCIE_EXERCISER_STATUS_SNAPSHOT_FUNCTION_ADDR + ( int )functionNum, 32, 0 );
}

//--------------------------------------------------------------------
void CAgtPCIEExerciser::ExerciserPhyStatusRead(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEExerciserPhyStatus status,
  /* [in] */ AgtSizeT lane_number,
  /* [retval][out] */ AgtValueT* val
  )
{

  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserPhyStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, lane_number,
    ParamType_EPCIEExerciserPhyStatus, status
    );

  TRACE("CAgtPCIEExerciser::ExerciserPhyStatusRead");

  return RegRead(portHandle, PCIE_EXERCISERPHYSTATUS_ADDR + (UInt32) status, (UInt8)lane_number, val);
}

//--------------------------------------------------------------------
void
CAgtPCIEExerciser::ExerciserChannelFunctionGet(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEExerciserHwChannelStatus  status,
  /* [retval][out] */ AgtValueT* val
  )
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserChannelFunctionGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEExerciserHwChannelStatus, status
    );

  TRACE("CAgtPCIEExerciser::ExerciserChannelFunctionGet");

  HRESULT hr = S_OK;

  return RegRead(portHandle, PCIE_EXERCISERCHANNELFUNCTIONGET_ADDR + (UInt32)status, functionNum,val); 
}

//--------------------------------------------------------------------
void
CAgtPCIEExerciser::ExerciserChannelFunctionSet(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEExerciserHwChannelStatus  status,
  /* [in] */ AgtValueT val
  )
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ExerciserChannelFunctionSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEExerciserHwChannelStatus, status
    );

  TRACE("CAgtPCIEExerciser::ExerciserChannelFunctionSet");

  HRESULT hr = S_OK;

  return RegWrite(portHandle,PCIE_EXERCISERCHANNELFUNCTIONSET_ADDR + (UInt32)status ,functionNum,  val);
}



#pragma endregion

#pragma region Virtual channel related

void CAgtPCIEExerciser::VCWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT vcNum,
  /* [in] */ EPCIEVC prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, vcNum,
    ParamType_EPCIEVC, prop,
    ParamType_AgtValueT, val);

  TRACE( "CAgtPCIEExerciser::VCWrite" );

  UInt32 vcBaseAddr = 0;

  switch( functionNum )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELA;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELB;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELC;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELD;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELE;
    break;
  default:
    // VCWrite is supported on 3 functions only
    AGT_THROW( "VCWrite: Parameter functionNum should be on of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );
  }

  return RegWrite( portHandle, vcBaseAddr + vcNum * PCIE_VCPROP_SIZE + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::VCRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ AgtSizeT vcNum,
  /* [in] */ EPCIEVC prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_AgtSizeT, vcNum,
    ParamType_EPCIEVC, prop );

  TRACE( "CAgtPCIEExerciser::VCRead" );

  UInt32 vcBaseAddr = 0;

  switch( functionNum )
  {
  case PCIE_HWCHANNEL_FUNCTION_A:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELA;
    break;
  case PCIE_HWCHANNEL_FUNCTION_B:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELB;
    break;
  case PCIE_HWCHANNEL_FUNCTION_C:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELC;
    break;
  case PCIE_HWCHANNEL_FUNCTION_D:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELD;
    break;
  case PCIE_HWCHANNEL_FUNCTION_E:
    vcBaseAddr = PCIE_VCPROP_ADDR_HWCHANNELE;
    break;
  default:
    // VCRead is supported on 3 functions only
    AGT_THROW( "VCRead: Parameter functionNum should be on of the following: PCIE_HWCHANNEL_FUNCTION_A, PCIE_HWCHANNEL_FUNCTION_B, PCIE_HWCHANNEL_FUNCTION_C, PCIE_HWCHANNEL_FUNCTION_D, PCIE_HWCHANNEL_FUNCTION_E." );
  }

  return RegRead( portHandle, vcBaseAddr + vcNum * PCIE_VCPROP_SIZE + ( UInt32 )prop, 32, val );
}

#pragma endregion

#pragma region Lane skew related

void CAgtPCIEExerciser::LaneSkewRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT lane,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "LaneSkewRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, lane );

  TRACE( "CAgtPCIEExerciser::LaneSkewRead" );

  return RegRead( portHandle, PCIE_LANESKEW2_ADDR + ( ( UInt32) lane ), 32, val );
}

void CAgtPCIEExerciser::LaneSkewWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT lane,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "LaneSkewWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, lane,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::LaneSkewWrite" );

  BLOCKMEMORYSTOPCHECK

    return RegWrite( portHandle, PCIE_LANESKEW2_ADDR + ( ( UInt32 ) lane ), 32, val );
}

#pragma endregion

#pragma region VC Resource related

void CAgtPCIEExerciser::VCResourceWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEVCResourceProp prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCResourceWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEVCResourceProp, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::VCResourceWrite" );

  BLOCKMEMORYSTOPCHECK

  // TODO: check invalid options depending on exerciser mode.

  UInt32 mult = 0;

  switch( channel )
  {
  case PCIE_VC_RESOURCE_0:
    mult = 0x0;
    break;
  case PCIE_VC_RESOURCE_1:
    mult = 0x01;
    break;
  case PCIE_VC_RESOURCE_2:
    mult = 0x02;
    break;
  case PCIE_VC_RESOURCE_3:
    mult = 0x03;
    break;
  case PCIE_VC_RESOURCE_4:
    mult = 0x04;
    break;
  case PCIE_VL_RESOURCE_0:
    mult = 0x05;
    break;
  default:
    AGT_THROW( "VCResourceWrite: Parameter channel is invalid." );
  }

  return RegWrite( portHandle, PCIE_HWCHANNEL_FUNCTION_PROP_ADDR +
    mult * PCIE_HWCHANNEL_FUNCTION_PROP_SIZE  + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::VCResourceRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEVCResourceProp prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCResourceRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEVCResourceProp, prop );

  TRACE( "CAgtPCIEExerciser::VCResourceRead" );

  UInt32 mult = 0;

  // TODO: check invalid options depending on exerciser mode.

  switch( channel )
  {
  case PCIE_VC_RESOURCE_0:
    mult = 0x0;
    break;
  case PCIE_VC_RESOURCE_1:
    mult = 0x01;
    break;
  case PCIE_VC_RESOURCE_2:
    mult = 0x02;
    break;
  case PCIE_VC_RESOURCE_3:
    mult = 0x03;
    break;
  case PCIE_VC_RESOURCE_4:
    mult = 0x04;
    break;
  case PCIE_VL_RESOURCE_0:
    mult = 0x05;
    break;
  default:
    AGT_THROW( "VCResourceRead: Parameter channel is invalid." );
  }

  return RegRead( portHandle, PCIE_HWCHANNEL_FUNCTION_PROP_ADDR +
    mult * PCIE_HWCHANNEL_FUNCTION_PROP_SIZE  + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::VCResourceStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEVCResourceStatusProp prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCResourceStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEVCResourceStatusProp, prop );

  TRACE( "CAgtPCIEExerciser::VCResourceStatusRead" );

  return RegRead( portHandle, PCIE_HWCHANNELSTATUS_PROP_ADDR +
    ( UInt32 )channel * PCIE_HWCHANNELSTATUS_PROP_SIZE + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::VCResourceStatusSnapshot
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCResourceStatusSnapshot",
    ParamType_AgtPortHandleT, portHandle);

  TRACE( "CAgtPCIEExerciser::VCResourceStatusSnapshot" );

  return RegWrite( portHandle, PCIE_HWCHANNEL_FUNCTION_STATUS_SNAPSHOT_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::VCResourceStatusArrayRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [out][in] */ AgtSizeT *pCount,
  /* [out] */ UInt8 *psaData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "VCResourceStatusArrayRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel );

  TRACE( "CAgtPCIEExerciser::VCResourceStatusArrayRead" );

  if( channel == PCIE_VC_RESOURCE_0 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_A_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
  else if( channel == PCIE_VC_RESOURCE_1 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_B_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
  else if( channel == PCIE_VC_RESOURCE_2 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_C_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
  else if( channel == PCIE_VC_RESOURCE_3 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_D_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
  else if( channel == PCIE_VC_RESOURCE_4 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_E_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
  else if( channel == PCIE_VL_RESOURCE_0 )
  {
    return MemRead( portHandle, PCIE_HWCHANNEL_STATUS_ARRAY_HWCHANNEL_F_START_ADDR,
      PCIE_HWCHANNEL_FUNCTION_STATUS_ARRAY_REAL_SIZE * 5, 8, pCount, psaData );
  }
}

#pragma endregion

#pragma region RxError related

void CAgtPCIEExerciser::RxErrorSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIERxError prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxErrorSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIERxError, prop,
    ParamType_AgtValueT, val);

  TRACE( "CAgtPCIEExerciser::RxErrorSet" );

  return RegWrite( portHandle, PCIE_RXERROR_ADDR + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::RxErrorGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIERxError prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxErrorGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIERxError, prop );

  TRACE( "CAgtPCIEExerciser::RxErrorGet" );

  return RegRead( portHandle, PCIE_RXERROR_ADDR + ( UInt32 )prop, 32, val );
}

void CAgtPCIEExerciser::RxErrorStart
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxErrorStart",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::RxErrorStart" );

  return RegWrite( portHandle, PCIE_RXERROR_CTRL_ADDR, 32, 1 );
}

void CAgtPCIEExerciser::RxErrorStop
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxErrorStop",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::RxErrorStop" );

  return RegWrite( portHandle, PCIE_RXERROR_CTRL_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::RxErrorStatusGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIERxErrorStatus prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "RxErrorStatusGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIERxErrorStatus, prop );

  TRACE( "CAgtPCIEExerciser::RxErrorStatusGet" );

  return RegRead( portHandle, PCIE_RXERRORSTATUS_ADDR + ( UInt32 )prop, 32, val );
}

#pragma endregion

#pragma region Pattern term related

void CAgtPCIEExerciser::PatternDefaultSet
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternDefaultSet",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PatternDefaultSet" );

  BLOCKMEMORYSTOPCHECK

  return RegWrite( portHandle, PCIE_PATTERN_DEFAULT_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::PatternMaskWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT term,
  /* [in] */ EPCIEPattern prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternMaskWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, term,
    ParamType_EPCIEPattern, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::PatternMaskWrite" );

  BLOCKMEMORYSTOPCHECK

  if( term >= 0 && term < PCIE_NUM_OF_PATTERN_TERMS )
  {
    return RegWrite( portHandle, ( term * PCIE_PATTERN_MASK_SIZE ) + PCIE_PATTERN_MASK_START_ADDR + ( UInt32 )prop,
      32, val );
  }
  else
  {
    AGT_THROW( "PatternMaskWrite: Invalid value for parameter term" );
  }
}

void CAgtPCIEExerciser::PatternMaskRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT term,
  /* [in] */ EPCIEPattern prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternMaskRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, term,
    ParamType_EPCIEPattern, prop );

  TRACE( "CAgtPCIEExerciser::PatternMaskRead" );

  if( term >= 0 && term < PCIE_NUM_OF_PATTERN_TERMS )
  {
    return RegRead( portHandle, ( term * PCIE_PATTERN_MASK_SIZE ) + PCIE_PATTERN_MASK_START_ADDR + ( UInt32 )prop,
      32, val );
  }
  else
  {
    AGT_THROW( "PatternMaskRead: Invalid value for parameter term" );
  }
}

void CAgtPCIEExerciser::PatternValueWrite
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT term,
  /* [in] */ EPCIEPattern prop,
  /* [in] */ AgtValueT val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternValueWrite",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, term,
    ParamType_EPCIEPattern, prop,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::PatternValueWrite" );

  BLOCKMEMORYSTOPCHECK

  if( term >= 0 && term < PCIE_NUM_OF_PATTERN_TERMS )
  {
    return RegWrite( portHandle, ( term * PCIE_PATTERN_VALUE_SIZE ) + PCIE_PATTERN_VALUE_START_ADDR + ( UInt32 )prop,
      32, val );
  }
  else
  {
    AGT_THROW( "PatternValueWrite: Invalid value for parameter term" );
  }
}

void CAgtPCIEExerciser::PatternValueRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT term,
  /* [in] */ EPCIEPattern prop,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternValueRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, term,
    ParamType_EPCIEPattern, prop );

  TRACE( "CAgtPCIEExerciser::PatternValueRead" );

  if( term >= 0 && term < PCIE_NUM_OF_PATTERN_TERMS )
  {
    return RegRead( portHandle,
      ( term * PCIE_PATTERN_VALUE_SIZE ) + PCIE_PATTERN_VALUE_START_ADDR + ( UInt32 )prop,
      32, val );
  }
  else
  {
    AGT_THROW( "PatternValueRead: Invalid value for parameter term" );
  }
}

void CAgtPCIEExerciser::PatternStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtSizeT term,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtSizeT, term );

  TRACE( "CAgtPCIEExerciser::PatternStatusRead" );

  if( term >= 0 && term < PCIE_NUM_OF_PATTERN_TERMS )
  {
    return RegRead( portHandle, PCIE_PATTERN_STATUS_START_ADDR + term, 32, val );
  }
  else
  {
    AGT_THROW( "PatternStatusRead: Invalid value for parameter term" );
  }
}

void CAgtPCIEExerciser::PatternStatusReset
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternStatusReset",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PatternStatusReset" );

  return RegWrite( portHandle, PCIE_PATTERN_STATUS_RESET_ADDR, 32, 1 );
}

void CAgtPCIEExerciser::PatternRun
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternRun",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PatternRun" );

  return RegWrite( portHandle, PCIE_PATTERN_RUNSTOP_ADDR, 32, 1 );
}

void CAgtPCIEExerciser::PatternStop
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PatternStop",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PatternStop" );

  return RegWrite( portHandle, PCIE_PATTERN_RUNSTOP_ADDR, 32, 0 );
}

#pragma endregion

#pragma region Performance Counters

void CAgtPCIEExerciser::PerformanceCounterStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEPerformanceCounterStatus prop,
  /* [in] */ AgtSizeT hi,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEPerformanceCounterStatus, prop,
    ParamType_AgtSizeT, hi );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterStatusRead" );

  UInt32 startAddress = 0;

  if( hi )
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_HI_START_ADDR;
  }
  else
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_LO_START_ADDR;
  }

  return RegRead( portHandle, startAddress + ( UInt32 )prop, 0, val );
}

void CAgtPCIEExerciser::PerformanceCounterVCStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEVCResourceId channel,
  /* [in] */ EPCIEPerformanceCounterVCStatus prop,
  /* [in] */ AgtSizeT hi,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterVCStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEVCResourceId, channel,
    ParamType_EPCIEPerformanceCounterVCStatus, prop,
    ParamType_AgtSizeT, hi );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterVCStatusRead" );

  AgtSizeT index = 0;
  UInt32 startAddress = 0;

  if( hi )
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_HI_START_ADDR;
  }
  else
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_LO_START_ADDR;
  }

  return RegRead( portHandle, startAddress + ( UInt32 )prop, channel, val );
}

void CAgtPCIEExerciser::PerformanceCounterFunctionStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [in] */ EPCIEPerformanceCounterFunctionStatus prop,
  /* [in] */ AgtSizeT hi,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterFunctionStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum,
    ParamType_EPCIEPerformanceCounterFunctionStatus, prop,
    ParamType_AgtSizeT, hi );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterFunctionStatusRead" );

  UInt32 startAddress = 0;

  if( hi )
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_HI_START_ADDR;
  }
  else
  {
    startAddress = PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_LO_START_ADDR;
  }

  return RegRead( portHandle, startAddress + ( UInt32 )prop, functionNum, val );
}

void CAgtPCIEExerciser::PerformanceCounterStatusReadAll
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterStatusReadAll",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterStatusReadAll" );

  // Do snapshot [Do not unlock] //////////////////////////////////////////////
  RegWrite( portHandle, PCIE_PERFORMANCE_COUNTER_STATUS_SNAPSHOT_ADDR, 32, false, true, 0 );
  /////////////////////////////////////////////////////////////////////////////

  int i = 0, j = 0, k = 0;

  // Read performance statistics values

  // Global values ////////////////////////////////////////////////////////////

  const int nNumGlobalCounterEnums = 8;
  EPCIEPerformanceCounterStatus globalCounterEnums[nNumGlobalCounterEnums];

  globalCounterEnums[0] = PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN;               // Index Lo( 0), Hi ( 1)
  globalCounterEnums[1] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_CODING_ERR;          // Index Lo( 2), Hi ( 3)
  globalCounterEnums[2] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_DISPARITY_ERR;       // Index Lo( 4), Hi ( 5)
  globalCounterEnums[3] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_128B_130B_ERR;       // Index Lo( 6), Hi ( 7)
  globalCounterEnums[4] = PCIE_PERFORMANCECOUNTERSTATUS_SUCCESSFUL_LINK_TRAININGS;  // Index Lo( 8), Hi ( 9)
  globalCounterEnums[5] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_ACK;                   // Index Lo(10), Hi (11)
  globalCounterEnums[6] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_NAK;                   // Index Lo(12), Hi (13)
  globalCounterEnums[7] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_TX_NAK;                // Index Lo(14), Hi (15)

  for( j = 0; j < nNumGlobalCounterEnums; j++ )
  {
    // Low DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_LO_START_ADDR +
      ( UInt32 )globalCounterEnums[j], 0, true, true, (val + i) );
    i++;
    // High DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_HI_START_ADDR +
      ( UInt32 )globalCounterEnums[j], 0, true, true, (val + i) );
    i++;
  }

  /////////////////////////////////////////////////////////////////////////////

  // VC related ///////////////////////////////////////////////////////////////

  // Index - PCIE_VC_RESOURCE_0 - Lo(16), Hi (17)
  // Index - PCIE_VC_RESOURCE_1 - Lo(18), Hi (19)
  // Index - PCIE_VC_RESOURCE_2 - Lo(20), Hi (21)
  // Index - PCIE_VC_RESOURCE_3 - Lo(22), Hi (23)
  // Index - PCIE_VC_RESOURCE_4 - Lo(24), Hi (25)
  // Index - PCIE_VL_RESOURCE_0 - Lo(26), Hi (27)

  for( j = 0; j < PCIE_VC_RESOURCE_END; j++ )
  {
    // Low DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_LO_START_ADDR +
      ( UInt32 )PCIE_PERFORMANCECOUNTERSTATUS_DLLP_FC, ( EPCIEVCResourceId )j, true, true, (val + i) );
    i++;
    // High DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_HI_START_ADDR +
      ( UInt32 )PCIE_PERFORMANCECOUNTERSTATUS_DLLP_FC, ( EPCIEVCResourceId )j, true, true, (val + i) );
    i++;
  }

  /////////////////////////////////////////////////////////////////////////////

  // Function Related /////////////////////////////////////////////////////////

  const int nNumFunctionCounterEnums = 4;

  EPCIEPerformanceCounterFunctionStatus functionCounterEnums[nNumFunctionCounterEnums];
                                                                      //             FnA        |       FnB        |       FnC        |       FnD        |       FnE        |       FnBVF1     |       FnBVF2     |       FnCVF1     |       FnCVF2     |       FnDVF1     |       FnDVF2     |       FnEVF1     |       FnEVF2     |
  functionCounterEnums[0] = PCIE_PERFORMANCECOUNTERSTATUS_TLP_NUM;    // Index Lo( 28), Hi( 29) | Lo( 30), Hi( 31) | Lo( 32), Hi( 33) | Lo( 34), Hi( 35) | Lo( 36), Hi( 37) | Lo( 38), Hi( 39) | Lo( 40), Hi( 41) | Lo( 42), Hi( 43) | Lo( 44), Hi( 45) | Lo( 46), Hi( 47) | Lo( 48), Hi( 49) | Lo( 50), Hi( 51) | Lo( 52), Hi( 53) |
  functionCounterEnums[1] = PCIE_PERFORMANCECOUNTERSTATUS_DW_NUM;     // Index Lo( 54), Hi( 55) | Lo( 56), Hi( 57) | Lo( 58), Hi( 59) | Lo( 60), Hi( 61) | Lo( 62), Hi( 63) | Lo( 64), Hi( 65) | Lo( 66), Hi( 67) | Lo( 68), Hi( 69) | Lo( 70), Hi( 71) | Lo( 72), Hi( 73) | Lo( 74), Hi( 75) | Lo( 76), Hi( 77) | Lo( 78), Hi( 79) | 
  functionCounterEnums[2] = PCIE_PERFORMANCECOUNTERSTATUS_TX_TLP_NUM; // Index Lo( 80), Hi( 81) | Lo( 82), Hi( 83) | Lo( 84), Hi( 85) | Lo( 86), Hi( 87) | Lo( 88), Hi( 89) | Lo( 90), Hi( 91) | Lo( 92), Hi( 93) | Lo( 94), Hi( 95) | Lo( 96), Hi( 97) | Lo( 98), Hi( 99) | Lo(100), Hi(101) | Lo(102), Hi(103) | Lo(104), Hi(105) |
  functionCounterEnums[3] = PCIE_PERFORMANCECOUNTERSTATUS_TX_DW_NUM;  // Index Lo(106), Hi(107) | Lo(108), Hi(109) | Lo(110), Hi(111) | Lo(112), Hi(113) | Lo(114), Hi(115) | Lo(116), Hi(117) | Lo(118), Hi(119) | Lo(120), Hi(121) | Lo(122), Hi(123) | Lo(124), Hi(125) | Lo(126), Hi(127) | Lo(128), Hi( 129)| Lo(130), Hi( 131)|

  for( j = 0; j < nNumFunctionCounterEnums; j++ )
  {
    for( k = 0; k < PCIE_HWCHANNEL_FUNCTION_END; k++ )
    {
      // Low DWORD
      RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_LO_START_ADDR +
        ( UInt32 )functionCounterEnums[j], (EPCIEHwChannelFunction)k, true, true, (val + i) );
      i++;
      // High DWORD
      RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_HI_START_ADDR +
        ( UInt32 )functionCounterEnums[j], (EPCIEHwChannelFunction)k, true, true, (val + i) );
      i++;
    }
  }

  // Release the lock
  myController->ReleaseFunctionLock( portHandle );
}

void CAgtPCIEExerciser::PerformanceCounterStatusReadFunctionAll
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHwChannelFunction functionNum,
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterStatusReadFunctionAll",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHwChannelFunction, functionNum );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterStatusReadFunctionAll" );

  // Do snapshot [Do not unlock] //////////////////////////////////////////////
  RegWrite( portHandle, PCIE_PERFORMANCE_COUNTER_STATUS_SNAPSHOT_ADDR, 32, false, true, 0 );
  /////////////////////////////////////////////////////////////////////////////

  int i = 0, j = 0;

  // Read performance statistics values

  // Global values ////////////////////////////////////////////////////////////

  const int nNumGlobalCounterEnums = 8;
  EPCIEPerformanceCounterStatus globalCounterEnums[nNumGlobalCounterEnums];

  globalCounterEnums[0] = PCIE_PERFORMANCECOUNTERSTATUS_INTERVAL_LEN;               // Index Lo( 0), Hi ( 1)
  globalCounterEnums[1] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_CODING_ERR;          // Index Lo( 2), Hi ( 3)
  globalCounterEnums[2] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_DISPARITY_ERR;       // Index Lo( 4), Hi ( 5)
  globalCounterEnums[3] = PCIE_PERFORMANCECOUNTERSTATUS_DLLPHY_128B_130B_ERR;       // Index Lo( 6), Hi ( 7)
  globalCounterEnums[4] = PCIE_PERFORMANCECOUNTERSTATUS_SUCCESSFUL_LINK_TRAININGS;  // Index Lo( 8), Hi ( 9)
  globalCounterEnums[5] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_ACK;                   // Index Lo(10), Hi (11)
  globalCounterEnums[6] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_NAK;                   // Index Lo(12), Hi (13)
  globalCounterEnums[7] = PCIE_PERFORMANCECOUNTERSTATUS_DLLP_TX_NAK;                // Index Lo(14), Hi (15)

  for( j = 0; j < nNumGlobalCounterEnums; j++ )
  {
    // Low DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_LO_START_ADDR +
      ( UInt32 )globalCounterEnums[j], 0, true, true, (val + i) );
    i++;
    // High DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_STATUS_PROP_HI_START_ADDR +
      ( UInt32 )globalCounterEnums[j], 0, true, true, (val + i) );
    i++;
  }

  /////////////////////////////////////////////////////////////////////////////

  // VC related ///////////////////////////////////////////////////////////////

  // Index - PCIE_VC_RESOURCE_0 - Lo(16), Hi (17)
  // Index - PCIE_VC_RESOURCE_1 - Lo(18), Hi (19)
  // Index - PCIE_VC_RESOURCE_2 - Lo(20), Hi (21)
  // Index - PCIE_VC_RESOURCE_3 - Lo(22), Hi (23)
  // Index - PCIE_VC_RESOURCE_4 - Lo(24), Hi (25)
  // Index - PCIE_VL_RESOURCE_0 - Lo(26), Hi (27)

  for( j = 0; j < PCIE_VC_RESOURCE_END; j++ )
  {
    // Low DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_LO_START_ADDR +
      ( UInt32 )PCIE_PERFORMANCECOUNTERSTATUS_DLLP_FC, ( EPCIEVCResourceId )j, true, true, (val + i) );
    i++;
    // High DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_VC_STATUS_PROP_HI_START_ADDR +
      ( UInt32 )PCIE_PERFORMANCECOUNTERSTATUS_DLLP_FC, ( EPCIEVCResourceId )j, true, true, (val + i) );
    i++;
  }

  /////////////////////////////////////////////////////////////////////////////

  // Function Related /////////////////////////////////////////////////////////

  const int nNumFunctionCounterEnums = 4;

  EPCIEPerformanceCounterFunctionStatus functionCounterEnums[nNumFunctionCounterEnums];
  functionCounterEnums[0] = PCIE_PERFORMANCECOUNTERSTATUS_TLP_NUM;    // Index Lo( 28), Hi( 29)
  functionCounterEnums[1] = PCIE_PERFORMANCECOUNTERSTATUS_DW_NUM;     // Index Lo( 30), Hi( 31)
  functionCounterEnums[2] = PCIE_PERFORMANCECOUNTERSTATUS_TX_TLP_NUM; // Index Lo( 32), Hi( 33)
  functionCounterEnums[3] = PCIE_PERFORMANCECOUNTERSTATUS_TX_DW_NUM;  // Index Lo( 34), Hi( 35)

  for( j = 0; j < nNumFunctionCounterEnums; j++ )
  {
    // Low DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_LO_START_ADDR +
      ( UInt32 )functionCounterEnums[j], functionNum, true, true, (val + i) );
    i++;
    // High DWORD
    RegRead( portHandle, PCIE_PERFORMANCE_COUNTER_GEN3_FUNCTION_STATUS_PROP_HI_START_ADDR +
      ( UInt32 )functionCounterEnums[j], functionNum, true, true, (val + i) );
    i++;
  }

  // Release the lock
  myController->ReleaseFunctionLock( portHandle );
}

void CAgtPCIEExerciser::PerformanceCounterStatusReset
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterStatusReset",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterStatusReset" );

  return RegWrite( portHandle, PCIE_PERFORMANCE_COUNTER_STATUS_RESET_ADDR, 32, 0 );
}

void CAgtPCIEExerciser::PerformanceCounterStatusSnapshot
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "PerformanceCounterStatusSnapshot",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::PerformanceCounterStatusSnapshot" );

  return RegWrite( portHandle, PCIE_PERFORMANCE_COUNTER_STATUS_SNAPSHOT_ADDR, 32, 0 );
}

#pragma endregion

#pragma region TriggerOut related

void CAgtPCIEExerciser::TriggerOutSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIETriggerOut src,
  /* [in] */ AgtValueT val // 1=enable 0=disable
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "TriggerOutSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIETriggerOut, src,
    ParamType_AgtValueT, val );

  TRACE( "CAgtPCIEExerciser::TriggerOutSet" );

  // Enables/disables the various triggerout sources
  UInt32 mask = ( UInt32 )src;

  // now find the bit pos
  UInt8 bitPos = 0x0;

  while( ( ( mask >> bitPos ) & 0x1 ) != 0x1 && bitPos <= 31 )
  {
    bitPos++;
  }

  return RegWrite( portHandle, PCIE_TRIGGEROUTSET_ADDR, bitPos, val );
}

void CAgtPCIEExerciser::TriggerOutGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIETriggerOut src,
  /* [retval][out] */ AgtValueT *val // 1=enable 0=disable
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "TriggerOutGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIETriggerOut, src );

  TRACE( "CAgtPCIEExerciser::TriggerOutGet" );

  // Enables/disables the various triggerout sources
  UInt32 mask = ( UInt32 )src;

  // now find the bit pos
  UInt8 bitPos = 0x0;

  while( ( ( mask >> bitPos ) & 0x1 ) != 0x1 && bitPos <= 31 )
  {
    bitPos++;
  }

  return RegRead( portHandle, PCIE_TRIGGEROUTSET_ADDR, bitPos, val );
}

void CAgtPCIEExerciser::TriggerOutStatusRead
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtValueT prop, // 1 = first 0 = accumulated error register
  /* [retval][out] */ AgtValueT *val
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "TriggerOutStatusRead",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtValueT, prop );

  // Read first/accumulated error register
  TRACE( "CAgtPCIEExerciser::TriggerOutStatusRead" );

  return RegRead( portHandle, PCIE_TRIGGEROUTSTATUSREAD_ADDR, ( UInt8 )prop, val );
}

void CAgtPCIEExerciser::TriggerOutStatusClear
(
  /* [in] */ AgtPortHandleT portHandle
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "TriggerOutStatusClear",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::TriggerOutStatusClear" );

  return RegWrite( portHandle, PCIE_TRIGGEROUTSTATUSCLEAR_ADDR, 0, 0 );
}

#pragma endregion

#pragma region Dllp related

void CAgtPCIEExerciser::DllpDataSend
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ AgtValueT dllpData
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpDataSend",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_AgtValueT, dllpData );

  TRACE("CAgtPCIEExerciser::DllpDataSend");

  return RegWrite(portHandle,PCIE_DLLP_SEND_ADDR,32,dllpData);
}

void CAgtPCIEExerciser::DllpStatusGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllpStatus status,
  /* [retval][out] */ AgtValueT *value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpStatusGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllpStatus, status );

  TRACE("CAgtPCIEExerciser::DllpStatusGet");

  return RegRead(portHandle,PCIE_DLLP_STATUS_ADDR + (UInt32) status, 32, value);
}

void CAgtPCIEExerciser::DllpStatusClear
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllpStatus status
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpStatusClear",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllpStatus, status);

  TRACE("CAgtPCIEExerciser::DllpStatusClear");

  return RegWrite(portHandle,PCIE_DLLP_STATUS_ADDR+(UInt32)status,32,0x0);
}

void CAgtPCIEExerciser::DllpControlSet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllpControl prop,
  /* [in] */ AgtValueT value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpControlSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllpControl, prop,
    ParamType_AgtValueT, value);

  TRACE("CAgtPCIEExerciser::DllpControlSet");

  return RegWrite( portHandle, PCIE_DLLP_CONTROL_ADDR + ( UInt32 )prop, 32, value );
}

void CAgtPCIEExerciser::DllpPropertyGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllp prop,
  /* [retval][out] */ AgtValueT *value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpPropertyGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllp, prop );

  TRACE("CAgtPCIEExerciser::DllpPropertyGet");

  return RegRead( portHandle, PCIE_DLLP_PROP_ADDR + ( UInt32 ) prop, 32, value );
}

void CAgtPCIEExerciser::DllpPropertySet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEDllp prop,
  /* [in] */ AgtValueT value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "DllpPropertySet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEDllp, prop,
    ParamType_AgtValueT, value);

  TRACE("CAgtPCIEExerciser::DllpPropertySet");

  return RegWrite(portHandle,PCIE_DLLP_PROP_ADDR+(UInt32)prop,32,value);
}

#pragma endregion

#pragma region Scratch pad related

void CAgtPCIEExerciser::ScratchPadRegisterGet(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEScratchPadRegister registerNum,
  /* [retval][out] */ AgtValueT *value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ScratchPadRegisterGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEScratchPadRegister, registerNum );

  if( value == NULL )
    AGT_THROW( "Invalid NULL pointer passed to ScratchPadRegisterGet()." );

  UInt32 cfgSpaceAddress = 0;

  switch( registerNum )
  {
  case PCIE_SCRATCH_PAD_REGISTER0:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_0;
    break;
  case PCIE_SCRATCH_PAD_REGISTER1:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_1;
    break;
  case PCIE_SCRATCH_PAD_REGISTER2:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_2;
    break;
  case PCIE_SCRATCH_PAD_REGISTER3:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_3;
    break;
  }

  myController->ISPRegReadWithFnLock( portHandle, cfgSpaceAddress * 4, *value );
}

void CAgtPCIEExerciser::ScratchPadRegisterSet(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEScratchPadRegister registerNum,
  /* [in] */ AgtValueT value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "ScratchPadRegisterSet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEScratchPadRegister, registerNum,
    ParamType_AgtValueT, value);

  UInt32 cfgSpaceAddress = 0;

  switch( registerNum )
  {
  case PCIE_SCRATCH_PAD_REGISTER0:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_0;
    break;
  case PCIE_SCRATCH_PAD_REGISTER1:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_1;
    break;
  case PCIE_SCRATCH_PAD_REGISTER2:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_2;
    break;
  case PCIE_SCRATCH_PAD_REGISTER3:
    cfgSpaceAddress = PCIE_CFGPORT_SCRATCH_PAD_3;
    break;
  }

  myController->ISPRegWriteWithFnLock( portHandle, cfgSpaceAddress * 4, value );
}

#pragma endregion

#pragma region Others

void CAgtPCIEExerciser::HostControllerVersionGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [in] */ EPCIEHostControllerVersion prop,
  /* [retval][out] */ AgtValueT *value
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "HostControllerVersionGet",
    ParamType_AgtPortHandleT, portHandle,
    ParamType_EPCIEHostControllerVersion, prop );

  TRACE( "CAgtPCIEExerciser::HostControllerVersionGet" );

  return RegRead( portHandle, PCIE_HOST_CONTROLLER_VERSION_ADDR + ( UInt32 )prop, 32, value );
}

void CAgtPCIEExerciser::SerialNumberGet
(
  /* [in] */ AgtPortHandleT portHandle,
  /* [retval][out] */ UInt8 *pSerialNumber
)
{
  CFunctionLogger functionLogger( myController->GetDebugLogger(), "SerialNumberGet",
    ParamType_AgtPortHandleT, portHandle );

  TRACE( "CAgtPCIEExerciser::SerialNumberGet" );

  UInt32 serialNumberArray[PCIE_SERIAL_NUMBER_STRING_SIZE];

  memset( serialNumberArray, 0x0, PCIE_SERIAL_NUMBER_STRING_SIZE * 4 );
  memset( pSerialNumber, 0x0, PCIE_SERIAL_NUMBER_STRING_SIZE * 4 );

  for( UInt8 i = 0; i < PCIE_SERIAL_NUMBER_STRING_SIZE; i++ )
  {
    RegRead( portHandle, PCIE_SERIAL_NUMBER_STRING_ADDR + ( ( UInt32 ) i), 32, serialNumberArray[i] );
  }

  memcpy( ( void* )pSerialNumber, ( void* )( serialNumberArray ), PCIE_SERIAL_NUMBER_STRING_SIZE * 4 );

  // Only 13 characters are used.
  pSerialNumber[PCIE_SERIAL_NUMBER_STRING_SIZE * 4 - 3] = 0x0;
  pSerialNumber[PCIE_SERIAL_NUMBER_STRING_SIZE * 4 - 2] = 0x0;
  pSerialNumber[PCIE_SERIAL_NUMBER_STRING_SIZE * 4 - 1] = 0x0;
}

#pragma endregion

#pragma endregion